<?php

namespace App\Http\Controllers\Admin\Reports;

use App\Http\Controllers\Admin\Controller;
use App\Libraries\Tabler;
use App\Models\Appointment;
use App\Models\Patient;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\Rule;

class PatientHistoryController extends Controller
{

    public function index(Request $request)
    {
        $selected_patient = '';
        $data['result'] = '';

        if ($request->method() == 'POST') {

            $rules = [];

            if (!is_admin()) {
                $rules['patient'] = ['required', Rule::in(allowed_patients())];
            } else {
                $rules['patient'] = ['required', 'exists:patients,id'];
            }

            $messages = [
                'patient.in' => 'Please check the patient field',
            ];

            $validated = $request->validate($rules, $messages);

            $table = $this->prepAppointmentsModel($request, $validated['patient']);

            $data = $table->initTable();

            // Override from Tabler
            $data['search_form_action'] = route('admin.reports.patient_history');

            $selected_patient = $validated['patient'];

            
        }


        $data['patients'] = Patient::allowedPatients()->orderBy('first_name')->orderBy('second_name')->orderBy('third_name')->orderBy('last_name')->get();

        $css = ['bs-select/bootstrap-select.min.css'];
        $js = ['bs-select/bootstrap-select.min.js'];


        $query_string = request()->query();

        $args = '';

        if ( $query_string ) {

            $args = '?';

            foreach ( $query_string as $key => $value ) {
                
                $args .= $key .'='. $value;

            }

        }

        $data['selected_patient'] = $selected_patient;

        $data['print_all'] = route('admin.reports.patient_history.print' , [ 'patient' => $selected_patient , 'selection' => 'all'] ) . $args;
        $data['print_current'] = route('admin.reports.patient_history.print', ['patient' => $selected_patient, 'selection' => 'current'] ) . $args;;
 
        $this->loadView('reports.patient_history', 'reports.patient_history.title', 'reports.patient_history', [], $data, $css, $js);
    }

    /*=================================================*/

    private function prepAppointmentsModel($request, $patient_id)
    {

        $model = new Appointment();

        $model = $model->allowedPatients();

        $select = [
            'ref_number',
            'appointments.id as id',
            'appointment_date',
            'start_time',
            'treatments.treatment AS treatment',
            'branches.branch as branch',
            //DB::raw('DATE_FORMAT(`appointment_date`, "%a") as ap_day'),
            DB::raw('DATE_FORMAT(`start_time`, "%h:%i %p") as start_time'),
            DB::raw('DATE_FORMAT(`checked_in_at`, "%h:%i %p") as checked_in_at'),
            DB::raw('DATE_FORMAT(`finished_at`, "%h:%i %p") as finished_at'),
            'reviews.rating as rating',
            //'reviews.review as review',
        ];

        $model = $model->join('treatments', 'appointments.treatment_id', '=', 'treatments.id');

        $model = $model->join('branches', 'appointments.branch_id', '=', 'branches.id');

        $model = $model->join('patients', 'appointments.patient_id', '=', 'patients.id');

        $model = $model->leftJoin('reviews', 'reviews.appointment_id', '=', 'appointments.id');

        $model = $model->where('appointments.postponed', '=', 0);

        $model = $model->where('appointments.patient_id', '=', $patient_id);

        $model = $model->orderBy('appointment_date')->orderBy('start_time');

        $th = ['ref_number', 'treatment', 'branch', 'appointment_date', 'start_time', 'checked_in_at', 'finished_at', 'rating'];

        $sortable = ['ref_number', 'appointment_date', 'treatment', 'branch', 'start_time', 'checked_in_at', 'finished_at', 'rating'];

        $searchable = ['ref_number', 'appointment_date', 'treatment', 'branch'];

        $table = new Tabler('admin.appointments', $model, $select, $th, $sortable, $searchable, $request, true);

        return $table;
    }

    /*=================================================*/
    public function print(Request $request, $patient, $selection)
    {

        $data = [];

        // Patient History
        if ($selection == 'all') {

            $model = new Appointment();

            $obj = $model->getPatientAppointments($patient);

            $th = ['ref_number', 'treatment', 'branch', 'appointment_date', 'start_time', 'checked_in_at', 'finished_at', 'rating'];

            // to remove th stdClass
            $data['rows'] = (array) json_decode(json_encode($obj->get()->toArray()), true);

            foreach ($th as $db_col) {

                $data['columns'][$db_col] = '<th>' . __('admin/appointments.columns.' . $db_col) . '</th>';
            }
        } else {

            $table = $this->prepAppointmentsModel($request, $patient);

            $result = $table->initTable();

            $rows = $result['result']->toArray();

            $data['rows'] = $rows['data'];

            $data['columns'] = $result['columns'];
        }



        $data['_page_title'] = __('admin/reports.patient_history.title');

        $data['patient'] = Patient::findOrFail($patient);

        echo view('admin.reports.printable.history', $data);
    }
}
