<?php

namespace App\Http\Controllers\Admin;

use App\Libraries\Tabler;
use App\Models\TreatmentCategory;
use App\Rules\AlphaSpace;
use App\Rules\ArEnAlphaNumSpace;
use App\Rules\GeneralText;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\Rule;

class TreatmentCategoriesController extends Controller
{
    public function index(Request $request)
    {
        $table = $this->getTreatmentCategoriesTable($request);

        $data = $table->initTable();

        $this->loadView('treatment_categories.index', 'treatment_categories.title', 'treatment_categories', [], $data);
    }
    //---------------------------------------------------------------------
    /**
     * Add new category
     *
     * @param Request $request
     * @return void
     */
    public function add(Request $request)
    {
        $upload_path = $this->uploadPath();

        if ($request->getMethod() == 'POST') {

            $validated = $request->validate([
                'category' => ['required', 'unique:treatment_categories,name', 'min:2', 'max:50', new GeneralText],
                'description' => ['nullable', 'max:5000', new GeneralText],
                'image' => ['nullable', 'mimes:png,jpg,jpeg', 'min:1', 'max:2048']
            ]);

            $category = [
                'name' => $validated['category'],
                'description' => $validated['description'],
            ];

            if (!array_key_exists('shown', $validated)) {
                $category['show_on_public'] = 0;
            }

            if ($request->file('image') && $request->file('image')->isValid()) {

                $file_name = time() . '.' . $request->file('image')->guessExtension();

                $request->file('image')->move($upload_path, $file_name);

                $category['image'] = $file_name;
            }

            echo '<pre>';
            print_r($category);
            echo '</pre>';
            exit;

            TreatmentCategory::create($category);

            return redirect()->route('admin.treatment_categories')->with('success', __('admin/common.msgs.success.add'));
        }

        $this->loadView('treatment_categories.add', 'treatment_categories.add_title', 'treatment_categories.add');
    }

    //---------------------------------------------------------------------
    /**
     * Edit existing category
     *
     * @param Request $request
     * @param TreatmentCategory $treatment_category
     * @return void
     */
    public function edit(Request $request, TreatmentCategory $treatment_category)
    {
        $upload_path = $this->uploadPath();

        if ($request->getMethod() == 'POST') {

            $validated = $request->validate([
                'category' => ['required', Rule::unique('treatment_categories', 'name')->ignore($treatment_category->id), 'min:2', 'max:50', new GeneralText],
                'description' => ['nullable', 'max:5000', new GeneralText],
                'image' => ['nullable', 'mimes:png,jpg,jpeg', 'min:1', 'max:2048']
            ]);


            $category = [
                'name' => $validated['category'],
                'description' => $validated['description'],
            ];

            if (empty($request->input('no-image'))) {

                $category['image'] = null;

                $this->deleteFile($treatment_category->image);
            }

            if (!array_key_exists('shown', $request->all())) {
                $category['show_on_public'] = 0;
            } else {
                $category['show_on_public'] = 1;
            }

            if ($request->file('image') && $request->file('image')->isValid()) {

                $file_name = time() . '.' . $request->file('image')->guessExtension();

                $this->deleteFile($treatment_category->image);

                $request->file('image')->move($upload_path, $file_name);

                $category['image'] = $file_name;
            }

            TreatmentCategory::where('id', $treatment_category->id)->update($category);

            return redirect()->route('admin.treatment_categories')->with('success', __('admin/common.msgs.success.edit'));
        }

        $data['category'] = $treatment_category;

        $this->loadView('treatment_categories.edit', 'treatment_categories.edit_title', 'treatment_categories.edit', $treatment_category, $data);
    }

    //--------------------------------------------------------------------------------

    public function delete(Request $request, TreatmentCategory $treatment_category)
    {

        if ($treatment_category->treatments && $treatment_category->treatments->count() > 0) {

            return redirect()->route('admin.treatment_categories')->with('danger', __('admin/common.msgs.error.delete'));
        }


        $this->deleteFile($treatment_category->image);


        $treatment_category->delete();

        return redirect()->route('admin.treatment_categories')->with('success', __('admin/common.msgs.success.delete'));
    }


    private function uploadPath()
    {
        // When deployed
        $upload_path = base_path('../') . '/' . config('admin.public_uploads_dir');

        // Localhost for testing
        $upload_path = public_path(config('admin.public_uploads_dir'));

        return $upload_path;
    }

    private function deleteFile($file)
    {

        $upload_path = $this->uploadPath();

        // remove old image
        if ($file && file_exists($upload_path . '/' . $file)) {
            unlink($upload_path . '/' . $file);
        }
    }
    /*==================================================*/
    private function getTreatmentCategoriesTable(Request $request)
    {
        $model = new TreatmentCategory();

        $select = ['treatment_categories.id', 'treatment_categories.name as category_title', DB::raw('count(*) as treatments_count')];

        $model = $model->join('treatments', 'treatments.category_id', '=', 'treatment_categories.id');
        $model = $model->groupBy('category_title');
        $model = $model->orderBy('treatments_count', 'DESC');
        $model = $model->orderBy('category_title', 'ASC');

        $route_key = 'admin.treatment_categories';

        $th = ['category_title', 'treatments_count'];

        $searchable = ['category_title'];
        $sortable = ['category_title', 'treatments_count'];

        $table = new Tabler($route_key, $model, $select, $th, $sortable, $searchable, $request, true);
        return $table;
    }

    /*=======================================================*/
    public function print(Request $request, $selection)
    {
        if ($selection == 'all') {

            $select = ['treatment_categories.id', 'treatment_categories.name as category_title', DB::raw('count(*) as treatments_count')];
           
            $obj = TreatmentCategory::select($select)
                ->join('treatments', 'treatments.category_id', '=', 'treatment_categories.id')
                ->groupBy('category_title')
                ->orderBy('treatments_count', 'DESC')
                ->orderBy('category_title', 'ASC');


            $th = ['category_title', 'treatments_count'];


            // to remove th stdClass
            $data['rows'] = (array) json_decode(json_encode($obj->get()->toArray()), true);

            foreach ($th as $db_col) {
                $data['columns'][$db_col] = '<th>' . __('admin/treatment_categories.columns.' . $db_col) . '</th>';
            }
        } else {

            $table = $this->getTreatmentCategoriesTable($request);

            $result = $table->initTable();

            $rows = $result['result']->toArray();

            $data['rows'] = $rows['data'];

            $data['columns'] = $result['columns'];
        }

        $data['_page_title'] = __('admin/treatment_categories.title');

        echo view('admin.printables.list', $data);
    }
    /*=======================================================*/
}
