<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;

class User extends Model
{
    use HasFactory;

    protected $fillable = [
        'first_name', 'last_name', 'mobile', 'email', 'password', 'role_id', 'is_blocked','device_token','sms_count','login_attempts'
    ];

    protected $hidden = ['password'];


    /**
     * Scope a query to filter by allowed branches only
     *
     * @param  \Illuminate\Database\Eloquent\Builder  $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeAllowedUsers($query)
    {
        return (!is_admin() && user_role() ) ? $query->whereIn('users.id', allowed_users())->where('is_admin',0) : $query->where('is_admin',0);
    }

    /*********************
     *  Relations
     *********************/

    /**
     * User ROle
     *
     * @return Illuminate\Database\Eloquent\Concerns\HasRelationships::belongsTo
     */
    public function role()
    {
        return $this->belongsTo(Role::class);
    }
    //-----------------------------------------------------------
    /**
     * User Appointments
     *
     * @return Illuminate\Database\Eloquent\Concerns\HasRelationships::belongsToMany
     */
    public function appointments()
    {
        return $this->belongsToMany(Appointment::class)->withPivot(['admission_date','admission_time','checkout_time']);
    }
    //-----------------------------------------------------------
    /**
     * User Branch (user to branch)
     *
     * @return Illuminate\Database\Eloquent\Concerns\HasRelationships::belongsToMany
     */
    public function branches()
    {
        return $this->belongsToMany(Branch::class);
    }
    //-----------------------------------------------------------
    /**
     * Complaints added by the user
     *
     * @return Illuminate\Database\Eloquent\Concerns\HasRelationships::hasMany
     */
    public function complaints()
    {
        return $this->hasMany(Complaint::class);
    }
    //-----------------------------------------------------------
    /**
     * Side effects messages
     *
     * @return Illuminate\Database\Eloquent\Concerns\HasRelationships::hasMany
     */
    public function complaintComments()
    {
        return $this->hasMany(ComplaintComments::class);
    }

    /**
     * Side effects messages
     *
     * @return Illuminate\Database\Eloquent\Concerns\HasRelationships::hasMany
     */
    public function patientComments()
    {
        return $this->hasMany(PatientComment::class);
    }
    //-----------------------------------------------------------

    public function logs(){
        
        return $this->hasMany(UserActivityLog::class);
    
    }
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    /*********************
     *  Attributes
     *********************/

    public function getFullNameAttribute()
    {

        return "{$this->first_name} {$this->last_name}";

    }

    public function selectFullName()
    {

        return DB::raw("CONCAT(users.first_name,' ',users.last_name) AS user_full_name");
    } 

}
