<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Admin\Controller;
use App\Models\Appointment;
use App\Models\Notification;
use App\Models\Patient;
use App\Models\Treatment;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\Rule;

class Dashboard extends Controller
{


    public function index(Request $request)
    {

        if (is_admin()) {

            $this->managerDashboard($request);
        } else {

            $role = user_role();

            if ($role) {

                switch ($role->dashboard) {
                    case 0: //default
                        //$this->defaultDashboard();
                        $this->receptionDashboard($request);
                        break;
                    case 1: //physician
                        $this->physicianDashboard($request);
                        break;
                    case 2: //reception
                        $this->receptionDashboard($request);
                        break;
                    case 3: //manager
                        $this->managerDashboard($request);
                        //$this->receptionDashboard($request);
                        break;
                    default:
                        $this->defaultDashboard();
                        break;
                }
            } else {
                $this->defaultDashboard($request);
            }
        }
    }
    //------------------------------------------------------------------------------------

    /**
     * Loads default dashboard
     *
     * @return void
     */
    private function defaultDashboard(Request $request)
    {
    }
    //------------------------------------------------------------------------------------
    /**
     * Load super admin dashboard
     *
     * @return void
     */
    private function adminDashboard(Request $request)
    {
    }
    //------------------------------------------------------------------------------------
    /**
     * Load physician dashboard
     *
     * @return void
     */
    private function physicianDashboard(Request $request)
    {

        $today = date('Y-m-d');

        $data['appointments'] = Appointment::where(['physician_id' => user_id(), 'appointment_date' => $today])->orderBy('appointment_date', 'DESC')->get();

        $this->loadView('dashboard.physician', 'dashboard.title', 'dashboard', [], $data);
    }

    //------------------------------------------------------------------------------------
    /**
     * Load Manager dashboard
     *
     * @return void
     */
    private function managerDashboard(Request $request)
    {
        // Branch patients count
        //--- Patients
        $data['cards'][] = [
            'count' => Patient::allowedPatients()->get()->count(),
            'icon'  => 'fas fa-procedures',
            'title' => __('admin/dashboard.patients_count'),
            'link' => route('admin.patients')
        ];
        //-- Appointments
        $data['cards'][] = [
            'count' => Appointment::allowedBranches()->get()->count(),
            'icon'  => 'far fa-calendar-alt',
            'title' => __('admin/dashboard.appointments_count'),
            'link' => route('admin.appointments')
        ];
        //-- Employees
        $data['cards'][] = [
            'count' => User::allowedUsers()->get()->count(),
            'icon'  => 'fas fa-users',
            'title' => __('admin/dashboard.employees_count'),
            'link' => route('admin.users')
        ];
        //-- Treatments
        $data['cards'][] = [
            'count' => Treatment::allowedTreatments()->get()->count(),
            'icon'  => 'fas fa-hand-holding-medical',
            'title' => __('admin/dashboard.treatments_count'),
            'link' => route('admin.treatments')
        ];

        //-------------------------------------------------------------------
        // Charts
        //-------------------------------------------------------------------

        // Patient By Age

        $data['chart_male'] = $this->getPatientsByAgeRange('m');

        $data['chart_female'] = $this->getPatientsByAgeRange('f');

        $data['patient_chart_labels'] = "'0-12','13-17','18-24','25-35','36-45','46-55','56-65','65+'";



        // Patients By Gender
        $data['total_male_count'] = Patient::allowedPatients()->where('gender', 'm')->get()->count();
        $data['total_female_count'] = Patient::allowedPatients()->where('gender', 'f')->get()->count();



        // Patients By Region
        $patients_by_region = Patient::allowedPatients()->select('regions.region as region_name', DB::raw('count(*) as total'))->join('regions', 'patients.region_id', '=', 'regions.id')->groupBy('region_name')->get();

        $data['regions'] = '';
        $data['patients_by_region_count'] = '';

        if ($patients_by_region) {

            foreach ($patients_by_region as $patient) {
                $data['regions'] .= "'" . $patient->region_name . "',";
                $data['patients_by_region_count'] .= "'" . $patient->total . "',";
            }
        }

        $data['regions'] = rtrim($data['regions'], ",");
        $data['patients_by_region_count'] = rtrim($data['patients_by_region_count'], ",");


        // Appointments by month

        $start = date('Y-m-d', strtotime('-6 months'));
        $end = date('Y-m-d', strtotime('+6 months'));

        $appointments_by_month = Appointment::allowedBranches()->select(DB::raw('count(id) as `total`'), DB::raw('YEAR(appointment_date) year, MONTH(appointment_date) as month'))
            ->whereBetween('appointment_date', [$start, $end])
            ->orderBy('appointment_date')
            ->groupBy('year', 'month')
            ->get();

        $data['month_labels'] = '';
        $data['app_by_month_counts'] = '';

        foreach ($appointments_by_month as $app) {
            $data['month_labels'] .= "'" . $app->month . "/" . $app->year . "',";
            $data['app_by_month_counts'] .= "'" . $app->total . "',";
        }

        $data['month_labels'] = rtrim($data['month_labels'], ",");
        $data['app_by_month_counts'] = rtrim($data['app_by_month_counts'], ",");

        //--------------------------------------------------------------------

        // Most 10 in-demand services in each branch
        $select = ['appointments.id', 'treatments.id as trtmnt_id', 'treatments.treatment as treatment_title', DB::raw('count(*) as total')];
        $result = Appointment::allowedBranches()->select($select)->join('treatments', 'appointments.treatment_id', '=', 'treatments.id')->groupBy('treatment_title')->get();

        $data['treatment_labels'] = '';
        $data['treatment_counts'] = '';

        if ($result) {

            foreach ($result as $treatment) {
                $data['treatment_labels'] .= "'" . $treatment->treatment_title . "',";
                $data['treatment_counts'] .= "'" . $treatment->total . "',";
            }
        }

        $data['treatment_labels'] = rtrim($data['treatment_labels'], ",");
        $data['treatment_counts'] = rtrim($data['treatment_counts'], ",");

        //--------------------------------------------------------------------

        // appointments by branch
        /*  $result = Appointment::allowedBranches()
                               ->select('appointments.id', 'branches.id as the_branch_id', 'branches.branch as branch_name', DB::raw('count(*) as total'))
                               ->join('branches', 'appointments.branch_id', '=', 'branches.id')
                               ->groupBy('branch_name')
                               ->get();

        
        
        $data['treatment_labels'] = '';
        $data['treatment_counts'] = '';
                       
        if ( $result ){
                       
            foreach ( $result as $treatment ) {
                $data['treatment_labels'] .= "'" . $treatment->treatment_title."',"; 
                $data['treatment_counts'] .= "'" . $treatment->total."',"; 
            }
                       
        }
                               
        $data['treatment_labels'] = rtrim( $data['treatment_labels'],",");
        $data['treatment_counts'] = rtrim( $data['treatment_counts'],","); */


        $js = ['chartjs/chart.min.js'];

        $this->loadView('dashboard.manager', 'dashboard.title', 'dashboard', [], $data, [], $js);
    }
    //------------------------------------------------------------------------------------

    /**
     * Load reception dashboard
     *
     * @return void
     */
    private function receptionDashboard(Request $request)
    {
        $today = date('Y-m-d');

        $model = new Appointment();

        $patient = $model->patientFullName();

        $model = $model->allowedPatients();

        $confirmed = __('admin/common.text.yes');
        $not_confirmed = __('admin/common.text.no');

        $status = DB::raw("IF(is_confirmed = 1,'" . $confirmed . "', '" . $not_confirmed . "') AS status");

        $select = [
            'ref_number', 'appointments.id as id', 'start_time', 'treatments.treatment AS treatment',
            $patient, $status
        ];

        $model = $model->join('treatments', 'appointments.treatment_id', '=', 'treatments.id');

        $model = $model->join('patients', 'appointments.patient_id', '=', 'patients.id');

        $model = $model->where('appointments.postponed', '=', 0);

        $model = $model->where('appointments.appointment_date', '=', $today);

        $model = $model->where('appointments.checked_in_at', '=', null);

        $model = $model->orderBy('start_time');

        //$data = $table->initTable();
        $data['appointments'] = $model->select($select)->get();

        $data['checked_in_appointments'] = $this->checkedInAppointments();

        $data['active_sessions'] = $this->getBusyPhysicians();

        $data['treatments'] = Treatment::allowedTreatments()->select(['treatments.id', 'treatment'])->orderBy('treatment')->get();

        $css = ['bs-select/bootstrap-select.min.css'];
        $js = ['bs-select/bootstrap-select.min.js'];

        $this->loadView('dashboard.reception', 'dashboard.title', 'dashboard', [], $data, $css, $js);
    }

    /****************************
     * Reception
     *****************************/
    //------------------------------------------------------------------------------------
    public function checkIn(Request $request, Appointment $appointment)
    {

        if (!is_admin() && !in_array($appointment->patient_id, allowed_patients())) {
            abort(404);
        }

        $now = now();

        $appointment->update(['checked_in_at' => $now]);

        return redirect()->route('admin.dashboard');
    }

    //------------------------------------------------------------------------------------
    public function admitPatient(Request $request, Appointment $appointment)
    {

        if ($request->method() == 'POST' && $request->ajax()) {

            if (!is_admin() && !in_array($appointment->patient_id, allowed_patients())) {
                abort(404);
            }

            $reserved_physicians = Appointment::allowedUsers()->where('appointment_date', date('Y-m-d'))->whereNotNull('admitted_at')->where('finished_at', null)->pluck('physician_id');

            $available_users = User::where('role_id', 1)->whereNotIn('id', $reserved_physicians)->orderBy('first_name')->get()->pluck('id');

            $validated = $request->validate([
                'free-physician' => ['required', Rule::in($available_users)]
            ]);

            if ($appointment->scheduled == 1) {

                $update = [
                    'admitted_at' => date('H:i'),
                    'physician_id' => $validated['free-physician'],
                ];
            } else {

                $now = date('H:i');
                $update = [
                    'admitted_at' => $now,
                    'physician_id' => $validated['free-physician'],
                    'start_time' => $now,
                    'end_time' => date('H:i', strtotime($now . ' +' . $appointment->treatment->session_period . ' minutes')),
                ];
            }

            $appointment->update($update);

            notify_user($validated['free-physician'], 'New Appointment', 'New appointment has been assigned', route('admin.dashboard'));

            return response()->json(['success' => 'success']);
        }
    }
    //------------------------------------------------------------------------------------

    public function endSession(Request $request, Appointment $appointment)
    {

        /*  if (!is_admin() && !in_array($appointment->patient_id, allowed_patients())) {
            abort(404);
        } */

        $appointment->update(['finished_at' => date('H:i')]);

        return redirect()->route('admin.dashboard');
    }
    //------------------------------------------------------------------------------------
    /**
     * Get Available physicians
     *
     * @param Request $request
     * @return void
     */
    public function getAvailablePhysicians(Request $request)
    {

        if ($request->ajax()) {

            $reserved_physicians = Appointment::allowedUsers()->where('appointment_date', date('Y-m-d'))->whereNotNull('admitted_at')->where('finished_at', null)->pluck('physician_id');

            $available_users = User::allowedUsers()->where('role_id', 1)->whereNotIn('id', $reserved_physicians)->orderBy('first_name')->get();

            return response()->json(['users' => $available_users]);
        }
    }
    //-----------------------------------------------------------------------------------
    public function getBusyPhysicians()
    {

        //if ( $request->ajax() ) {

        $active_sessions = Appointment::allowedPatients()->allowedUsers()
            ->where('appointment_date', date('Y-m-d'))
            ->whereNotNull('admitted_at')
            ->where('finished_at', null)
            ->get();

        return $active_sessions;

        //}
    }
    //------------------------------------------------------------------------------------
    /**
     * Get patients without appointments on that day
     *
     * @param Request $request
     * @return void
     */
    public function getNoAppPatients(Request $request)
    {

        $search_query = $request->input('sq');

        $today = date('Y-m-d');

        $patients_with_appointments = Appointment::allowedPatients()->where(['appointment_date' => $today])->pluck('patient_id');

        $patients = Patient::allowedPatients()->whereNotIn('id', $patients_with_appointments)->where('mobile', 'like', '%' . $search_query . '%')->get();

        return response()->json(['patients' => $patients], 200);
    }
    //------------------------------------------------------------------------------------
    public function addPatientNoApp(Request $request)
    {

        if ($request->method() == 'POST') {

            $validated = $request->validate([
                'no-app-patient' => ['required', Rule::in(allowed_patients())],
                'no-app-treatment' => ['required', Rule::in(allowed_treatments())],
            ]);


            $branches = allowed_branches();

            $year = date('y');
            $month = date('m');
            $day = date('d');
            $hour = date('Hi');

            $branch = $branches[0];
            $treatment_id =  $validated['no-app-treatment'];
            $patient_id =  $validated['no-app-patient'];

            $ref_number = $year . $month . $day . $hour . $branches[0] . $treatment_id . $patient_id . '0';

            $appointment = [
                'ref_number' => $ref_number,
                'branch_id' => $branch,
                'treatment_id' => $treatment_id,
                'patient_id' => $patient_id,
                'appointment_date' => date('Y-m-d'),
                'start_time' => null,
                'end_time' => null,
                'checked_in_at' => now(),
                'scheduled' => 0
            ];

            Appointment::create($appointment);


            //return response()->json(['success' => 'success'],200);

            return redirect()->route('admin.dashboard');
        }
    }

    //------------------------------------------------------------------------------------
    private function checkedInAppointments()
    {

        $appointments = Appointment::allowedPatients()->whereNotNull('checked_in_at')->where('admitted_at', null)->where('appointment_date', date('Y-m-d'))->get();

        return $appointments;
    }
    //------------------------------------------------------------------------------------
    /********************************************************/


    /*****************
     *  Notifications
     *****************/
    public function getNotifications(Request $request)
    {

        $response = [];

        if ($request->ajax()) {

            $notifications = user_notifications();

            if ($notifications) {
                $response = [
                    'notifications' => $notifications,
                    'count' => user_notifications_count()
                ];
            }
        }

        return response()->json($response, 200);
    }

    public function markAsRead(Request $request, $notification_id)
    {

        Notification::where(['sent_to' => user_id(), 'id' => $notification_id])->update(['read_at' => now()]);
    }

    private function getPatientsByAgeRange($gender = 'f', $as_string = true)
    {
        // Charts
        $ranges = [ // the start of each age-range.
            '0-12'  => 0,
            '13-17' => 13,
            '18-24' => 18,
            '25-35' => 25,
            '36-45' => 36,
            '46-55' => 46,
            '56-65' => 56,
            '65+' => 65,
        ];

        $result = Patient::allowedPatients()->where('gender', $gender)->get()->map(function ($patient) use ($ranges) {

            $age = Carbon::parse($patient->dob)->age;

            foreach ($ranges as $key => $breakpoint) {

                if ($breakpoint >= $age) {

                    $patient->range = $key;

                    break;
                }
            }

            return $patient;
        })
            ->mapToGroups(function ($patient, $key) {
                return [$patient->range => $patient];
            })
            ->map(function ($group) {
                return count($group);
            })
            ->sortKeys();

        $result_array = $result->toArray();

        // fill range gaps

        foreach ($ranges as $range => $value) {

            if (!array_key_exists($range, $result_array)) {
                $result_array[$range] = 0;
            }
        }
        // Sort array in asc order by key value
        ksort($result_array);




        if ($as_string) {
            return "'" . implode("','", $result_array) . "'";
        }

        return ksort($result_array);
    }
}
