<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Admin\Controller;
use App\Libraries\Tabler;
use App\Models\Patient;
use App\Models\PatientPretest;
use App\Models\Pretest;
use App\Models\User;
use App\Rules\GeneralText;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;

class PretestsController extends Controller
{
    private $response;

    //-------------------------------------------------------
    public function index(Request $request)
    {
        $table = $this->getPretestsTable($request);

        $data = $table->initTable();

        $this->loadView('pretests.index', 'pretests.title', 'pretests', [], $data);
    }
    //--------------------------------------------------------
    public function add(Request $request)
    {
        if ($request->method() == 'POST' && $request->ajax()) {

            $min_date = date('Y-m-d', strtotime("-5 years"));
            $max_date = date('Y-m-d');

            $validator = Validator::make($request->all(), [
                'pretest' => [
                    'required', 
                    'exists:pretests,id',
                    Rule::unique('patient_pretest','pretest_id')->where(function ($query) use($request) {
                        return $query->where('patient_id', $request->input('patient'))
                                     ->where('pretest_id', $request->input('pretest'))
                                     ->where('form_date',$request->input('form-date'));
                    }),
                ],
                'form-date' => ['required','date_format:Y-m-d', 'before_or_equal:' . $max_date, 'after_or_equal:' . $min_date],
            ]);

            if ( !is_admin() ){
                $rules['patient'] = ['required', 'exists:patients,id',Rule::in(allowed_patients())];
            }else{
                $rules['patient'] = ['required', 'exists:patients,id'];
            }

            if ($validator->fails()) {

                $errorString = implode("<br>", $validator->errors()->all());
                return response()->json(['error' => $errorString], 400);
            }

            $pretest_id = $request->input('pretest');
            $patient_id = $request->input('patient');

             // if form has prerequisite
             if ( $this->hasPrerequisite($patient_id,$pretest_id) ) {
                return $this->response;
            }

        
            $this->validatePretestForm($request, $patient_id, $pretest_id);

            if ( $this->response ) {
                return $this->response;
            }

            $pretest = Pretest::findOrFail($pretest_id);

            $form = $request->all();
            unset($form['_token']);
            unset($form['pretest']);
            unset($form['patient']);
            unset($form['credibility']);


            // check if attachment is present

            if ($request->file('attachment') && $request->file('attachment')->isValid()) {

                $patient_dir = config('admin.pretest_attachments_dir') . '/' . $patient_id;

                $file_name = time() . '.' . $request->file('attachment')->guessExtension();

                $request->file('attachment')->storeAs($patient_dir, $file_name);

                $form['attachment'] = $file_name;
            }

            // change title

            // Laser Session for physician
            if ( $pretest_id == 4 ) {

                if (array_key_exists('session-number', $form) && array_key_exists('session-type', $form)) {

                    $session_type = ($form['session-type'] == 'c') ? __('pretests.laser_sessions.types.control') :  __('pretests.laser_sessions.types.control');
        
                    $number = $form['session-number'];
        
                    $title = $pretest->title . ' - ' . $session_type . ' - ' . $number;
                } 
            }
            $title = $pretest->title;

            $pretest_data = [
                'patient_id' =>  $patient_id,
                'pretest_id' =>  $pretest_id,
                'created_by' =>  user_id(),
                'title' => $title,
                'form_date' => $request->input('form-date'),
                'form' =>  json_encode($form, JSON_UNESCAPED_UNICODE),
            ];

            PatientPretest::create($pretest_data);

            session()->flash('success', __('admin/common.msgs.success.add'));
            return response()->json(['success' => 'success'], 200);
        }

        $data['patients'] = Patient::orderBy('first_name')->orderBy('second_name')->orderBy('third_name')->orderBy('last_name')->get();

        $data['pretests'] = Pretest::orderBy('title')->get();


        $css = ['bs-select/bootstrap-select.min.css','bs-datepicker/dp-bs-4.min.css'];

        $js = ['bs-select/bootstrap-select.min.js','momentjs/moment.min.js', 'bs-datepicker/dp-bs-4.min.js'];

        $this->loadView('pretests.add', 'pretests.add_title', 'pretests.add', [], $data, $css, $js);
    }
    //--------------------------------------------------------
    public function edit(Request $request,PatientPretest $patient_pretest)
    {

        if ( !is_admin() && $patient_pretest->created_by != user_id() ){

            return ( $request->ajax() ) ?  response()->json(['error' => '401'], 401) : redirect()->back()->with('danger','التعديل على النموذج من خلال الأخصائي المسؤول فقط');
            
        }

        if ($request->method() == 'POST' && $request->ajax()) {
            
            $min_date = date('Y-m-d', strtotime("-5 years"));
            $max_date = date('Y-m-d');

            $validator = Validator::make($request->all(), [
                Rule::unique('patient_pretest','pretest_id')->where(function ($query) use($request) {
                    return $query->where('patient_id', $request->input('patient'))
                                 ->where('pretest_id', $request->input('pretest'))
                                 ->where('form_date',$request->input('form-date'));
                })->ignore($patient_pretest->id),
                'form-date' => ['required','date_format:Y-m-d', 'before_or_equal:' . $max_date, 'after_or_equal:' . $min_date]
            ]);

            if ( !is_admin() ){
                $rules['patient'] = ['required', 'exists:patients,id',Rule::in(allowed_patients())];
            }else{
                $rules['patient'] = ['required', 'exists:patients,id'];
            }

            if ($validator->fails()) {

                $errorString = implode("<br>", $validator->errors()->all());
                return response()->json(['error' => $errorString], 400);
            }

            $pretest_id = $request->input('pretest');
            $patient_id = $request->input('patient');

             // if form has prerequisite
             if ( $this->hasPrerequisite($patient_id,$pretest_id) == true ) {

                $prerequisite_pretest = Pretest::findOrFail($patient_pretest->prerequisite);
    
                $title = $prerequisite_pretest->title;
        
                $data['info'] = sprintf(__('pretests.errors.prerequisite'),$title);
        
                $this->response = $data['info'];

                return $this->response;
            }

        
            $this->validatePretestForm($request, $patient_id, $pretest_id);

            if ($this->response) {
                return $this->response;
            }

            $pretest = Pretest::findOrFail($pretest_id);

            $form = $request->all();
            unset($form['_token']);
            unset($form['pretest']);
            unset($form['patient']);
            unset($form['credibility']);


             // check if attachment is present

             if ($request->file('attachment') && $request->file('attachment')->isValid()) {

                $patient_dir = config('admin.pretest_attachments_dir') . '/' . $patient_id;

                $file_name = time() . '.' . $request->file('attachment')->guessExtension();

                $request->file('attachment')->storeAs($patient_dir, $file_name);

                $form['attachment'] = $file_name;
            
            }elseif( array_key_exists('uploaded-file',$request->all() ) ){

                $form['attachment'] = $request->input('uploaded_file');

            }


            // change title
            // Laser Session for physician
            if ( $pretest_id == 4 ) {

                if (array_key_exists('session-number', $form) && array_key_exists('session-type', $form)) {

                    $session_type = ($form['session-type'] == 'c') ? __('pretests.laser_sessions.types.control') :  __('pretests.laser_sessions.types.control');
        
                    $number = $form['session-number'];
        
                    $title = $pretest->title . ' - ' . $session_type . ' - ' . $number;
                } 
            }
            $title = $pretest->title;

            $pretest_data = [
                'patient_id' =>  $patient_id,
                'pretest_id' =>  $pretest_id,
                'created_by' =>  user_id(),
                'title' => $title,
                'form_date' => $request->input('form-date'),
                'form' =>  json_encode($form, JSON_UNESCAPED_UNICODE),
            ];

            $patient_pretest->update($pretest_data);

            session()->flash('success', __('admin/common.msgs.success.edit'));
            return response()->json(['success' => 'success'], 200);
        }

        $data['patient'] = Patient::findOrFail($patient_pretest->patient_id);

        $data['patients'] = Patient::orderBy('first_name')->orderBy('second_name')->orderBy('third_name')->orderBy('last_name')->get();

        $data['pretests'] = Pretest::orderBy('title')->get();

        $data['patient_pretest'] = $patient_pretest;

        $data['answers'] = json_decode($patient_pretest->form, true);

        $pretest = Pretest::findOrFail($patient_pretest->pretest_id);

        $data['info'] = '' ;
        
        $data['form'] = view('pretests.'.$pretest->view_file .'_edit',$data);

        $css = ['bs-select/bootstrap-select.min.css','bs-datepicker/dp-bs-4.min.css'];

        $js = ['bs-select/bootstrap-select.min.js','momentjs/moment.min.js', 'bs-datepicker/dp-bs-4.min.js'];

        $this->loadView('pretests.edit', 'pretests.edit_title', 'pretests.edit', $patient_pretest, $data, $css,$js);
    }
    //--------------------------------------------------------
    public function info(Request $request, $pretest_id){

        $pretest = PatientPretest::findOrFail($pretest_id);

        if ( !is_admin() && !in_array($pretest->patient_id,allowed_patients()) ){
            abort(404);
        }
        
    
        $patient = Patient::findOrFail($pretest->patient_id);

        $form = Pretest::findOrFail($pretest->pretest_id);
    
        $data['patient'] = $patient;
        $data['answers'] = json_decode($pretest->form, true);;
        
        return view('pretests.'.$form->view_file.'_readonly',$data);
    
    }
    //--------------------------------------------------------
/*     public function downloadAttachment(Request $request, $attachment)
    {
      
        $path = config('admin.pretest_attachments_dir') . '/' . $complaint->patient_id . '/' . $complaint->attachment;

        if (Storage::exists($path)) {

            return Storage::download($path);
        }

        abort(404);
    } */
    //--------------------------------------------------------
    public function getForm(Request $request, $patient_id, $pretest_id)
    {
        if ($request->ajax()) {

            $patient = Patient::findOrFail($patient_id);

            //-------------------------------------------------
            $pretest = Pretest::findOrFail($pretest_id);

            $info = '';

            if ( $this->hasPrerequisite($patient_id,$pretest_id) == true ){
                
                $prerequisite_pretest = Pretest::findOrFail($pretest->prerequisite);

                $title = $prerequisite_pretest->title;
    
                $info = sprintf(__('pretests.errors.prerequisite'),$title);
    
            }
  
            return response(view('pretests.' . $pretest->view_file, compact(['patient', 'pretest', 'info'])), 200);
        }
    }
    //---------------------------------------------------
    public function delete(Request $request, PatientPretest $patient_pretest){

        //TODO If patient is in the allowed patients then delete

        if ( !is_admin() && !in_array($patient_pretest->patient_id,allowed_patients()) ){
            abort(404);
        }
        

        $patient_pretest->delete();

        return redirect()->route('admin.pretests')->with('success', __('admin/common.msgs.success.delete'));

    }
    //---------------------------------------------------
    private function validatePretestForm(Request $request, $patient_id, $pretest_id)
    {

        $patient = Patient::findOrFail($patient_id);

        // Initialize the arrays
        $rules = [];
        $messages = [
            'required' => __('pretests.errors.required'),
            'required_if' => __('pretests.errors.required_if'),
        ];

        // get the appropriate rules
        switch ($pretest_id) {
            case 1:
                // General Form
                $rules = [
                    'q1' => ['required', Rule::in(['0', '1'])],
                    'q2' => ['required', Rule::in(['0', '1'])],
                    'q2-input' => ['required_if:q2,==,1', new GeneralText],
                    'q3' => ['required', Rule::in(['0', '1'])],
                    'q3-input' => ['required_if:q3,==,1', new GeneralText],
                    'q4' => ['required', Rule::in(['0', '1'])],
                    'q4-input' => ['required_if:q4,==,1', new GeneralText],
                    'q5' => [Rule::requiredIf($patient->gender == 'f'), Rule::in(['0', '1'])],
                    'q5-input' => ['required_if:q5,==,1', new GeneralText],
                    'q6' => ['nullable', new GeneralText],
                    'notes' => ['nullable', new GeneralText],
                ];
                break;
            case 2:
                // laser removal - Public
                $locations = __('pretests.laser_sessions.locations');
                $rules = [
                    'q1' => ['required', new GeneralText],
                    //'q2' => ['required', Rule::in(array_keys($locations))],
                    'q2' => ['required', new GeneralText],
                    'q3' => ['required', new GeneralText],
                    'q4' => ['required', new GeneralText],
                    'q5' => ['required', new GeneralText],
                    'q6' => ['required', new GeneralText],
                    'notes' => ['nullable', new GeneralText],
                ];
                break;
            case 3:
                $q1_answers = (array) __('pretests.skin_care_answers.q1');
                $q9_answers = (array) __('pretests.skin_care_answers.q9');

                $rules = [
                    'q1' => ['required', Rule::in(array_keys($q1_answers))],
                    'q2' => [Rule::requiredIf($patient->gender == 'f'), new GeneralText],
                    'q3' => [Rule::requiredIf($patient->gender == 'f'), Rule::in(['0', '1'])],
                    'q3-input' => ['required_if:q3,==,0', new GeneralText],
                    'q4' => ['required', Rule::in(['0', '1'])],
                    'q4-input' => ['required_if:q4,==,1', new GeneralText],
                    'q5' => ['required', new GeneralText],
                    'q6' => ['required', Rule::in(['0', '1'])],
                    'q6-input' => ['required_if:q6,==,1', new GeneralText],
                    'q7' => ['required', new GeneralText],
                    'q8' => ['required', new GeneralText],
                    'q9' => ['required', Rule::in(array_keys($q9_answers))],
                    'q9-input' => ['nullable', new GeneralText],
                    'q10' => [Rule::requiredIf(user_id()), new GeneralText],
                    'notes' => ['nullable', new GeneralText],
                    'attachment' => ['mimes:png,jpg,jpeg', 'min:10', 'max:5120']
                ];
                break;
            case 4:
                // Laser sessions - Physician Only
                $min_date = date('Y-m-d', strtotime("-5 years"));
                $max_date = date('Y-m-d');
                $rules = [
                    'session-number' => ['required', 'integer', 'gte:1', 'lte:20'],
                    'session-date' => ['required', 'date_format:Y-m-d', 'before_or_equal:' . $max_date, 'after_or_equal:' . $min_date],
                    'session-type' => ['required', Rule::in(['c', 's'])],
                    'body-spot-size' => ['nullable', new GeneralText],
                    'body-pulse' => ['nullable', new GeneralText],
                    'body-e' => ['nullable', new GeneralText],
                    'body-hair' => ['nullable', new GeneralText],
                    'face-spot-size' => ['nullable', new GeneralText],
                    'face-pulse' => ['nullable', new GeneralText],
                    'face-e' => ['nullable', new GeneralText],
                    'face-hair' => ['nullable', new GeneralText],
                    'sens-spot-size' => ['nullable', new GeneralText],
                    'sens-pulse' => ['nullable', new GeneralText],
                    'sens-e' => ['nullable', new GeneralText],
                    'sens-hair' => ['nullable', new GeneralText],
                    'notes' => ['nullable', new GeneralText],
                ];
                break;
            case 5:
                // Fractional statement (public)
                $rules = [
                    'q1' => ['required', Rule::in(['0', '1'])],
                    'q2' => ['required', Rule::in(['0', '1'])],
                    'q2-input' => ['required_if:q2,==,1', new GeneralText],
                    'q3' => ['required', Rule::in(['0', '1'])],
                    'q3-input' => ['required_if:q3,==,1', new GeneralText],
                    'q4' => ['required', Rule::in(['0', '1'])],
                    'q4-input' => ['required_if:q4,==,1', new GeneralText],
                    'q5' => [Rule::requiredIf($patient->gender == 'f'), Rule::in(['0', '1'])],
                    'q5-input' => ['required_if:q5,==,1', new GeneralText],
                    'q6' => ['nullable', new GeneralText],
                    'notes' => ['nullable', new GeneralText],
                ];
                break;
            case 6:
                // Fractional sessions (physician)
                $min_date = date('Y-m-d', strtotime("-5 years"));
                $max_date = date('Y-m-d');

                $rules = [
                    'session-number' => ['required', 'integer', 'gte:1', 'lte:20'],
                    'session-date' => ['required', 'date_format:Y-m-d', 'before_or_equal:' . $max_date, 'after_or_equal:' . $min_date],
                    'physician' => ['required', Rule::in([user_name()])],
                    'light' => ['nullable', new GeneralText],
                    'mid' => ['nullable', new GeneralText],
                    'deep' => ['nullable', new GeneralText],
                    'note' => ['nullable', new GeneralText],
                    'notes' => ['nullable', new GeneralText],
                ];
                break;
            default:
                $rules = [
                    'q1' => ['required', Rule::in(['0', '1'])],
                    'q2' => ['required', Rule::in(['0', '1'])],
                    'q2-input' => ['required_if:q2,==,1', new GeneralText],
                    'q3' => ['required', Rule::in(['0', '1'])],
                    'q3-input' => ['required_if:q3,==,1', new GeneralText],
                    'q4' => ['required', Rule::in(['0', '1'])],
                    'q4-input' => ['required_if:q4,==,1', new GeneralText],
                    'q5' => [Rule::requiredIf($patient->gender == 'f'), Rule::in(['0', '1'])],
                    'q5-input' => ['required_if:q5,==,1', new GeneralText],
                    'q6' => ['nullable', new GeneralText],
                    'notes' => ['nullable', new GeneralText],
                ];
                break;
        }

        $validator = Validator::make($request->all(), $rules, $messages);

        if ($validator->fails()) {

            $errorString = implode("<br>", $validator->errors()->all());
            $this->response = response()->json(['error' => $errorString], 400);

            return false;
        }

        return true;
    }
    //---------------------------------------------------
    private function hasPrerequisite($patient_id, $pretest_id)
    {

        $pretest = Pretest::findOrFail($pretest_id);

        $prerequisite = $pretest->prerequisite;

        if ( $prerequisite == 0 ) {
            return false;
        }

        $result = PatientPretest::where('patient_id', $patient_id)->where('pretest_id', $prerequisite)->get();

        if ($result && $result->count() < 1) {

            $prerequisite_pretest = Pretest::findOrFail($prerequisite);

            $title = $prerequisite_pretest->title;

            $error = sprintf(__('pretests.errors.prerequisite'),$title);

            $this->response  = response()->json(['error' => $error],400);
            
            return true;
        }

        return false;
    }

    /*===================================================================*/

    private function getPretestsTable(Request $request){

        $model = new PatientPretest();

        $patient = new Patient();

        $patient_name = $patient->patientFullName();

        $user = new User();

        $user_full_name = $user->selectFullName();

        //$select_columns = ['patient_pretest.id', DB::raw('DATE_FORMAT(patient_pretest.created_at, "%Y-%m-%d") as form_date'), 'patient_pretest.title', $patient_name, $user_full_name];
        $select_columns = ['patient_pretest.id', 'form_date', 'patient_pretest.title', $patient_name, $user_full_name];

        $model = $model->join('patients', 'patient_pretest.patient_id', '=', 'patients.id');
        $model = $model->join('pretests', 'patient_pretest.pretest_id', '=', 'pretests.id');
        $model = $model->join('users', 'patient_pretest.created_by', '=', 'users.id');
        
        $model = $model->allowedPatients();

        $th_columns = ['form_date', 'title', 'patient_name', 'user_full_name'];

        $sortable_columns = ['patient_pretest.title', 'form_date', 'patient_name', 'user_full_name'];
        $searchable_columns = ['patient_pretest.title', 'form_date', 'patient_name', 'user_full_name'];

        $route_key = 'admin.pretests';

        $table = new Tabler($route_key, $model, $select_columns, $th_columns, $sortable_columns, $searchable_columns, $request,true);

        return $table;
    }


    /*===================================================================*/
    public function downloadAttachment(Request $request, $pid, $attachment)
    {
        if ( !is_admin() && !in_array($pid,allowed_patients()) ){
            abort(404);
        }

        $path = config('admin.pretest_attachments_dir') . '/' . $pid . '/' . $attachment;

        if (Storage::exists($path)) {

            return Storage::download($path);
        }

        abort(404);
    }


    /*===================================================================*/

    public function print(Request $request, $selection)
    {

        if ($selection == 'all') {

            $patient = new Patient();
    
            $patient_name = $patient->patientFullName();
    
            $user = new User();
    
            $user_full_name = $user->selectFullName();
    
            //$select_columns = ['patient_pretest.id', DB::raw('DATE_FORMAT(patient_pretest.created_at, "%Y-%m-%d") as form_date'), 'patient_pretest.title', $patient_name, $user_full_name];
            $select_columns = ['patient_pretest.id', 'form_date', 'patient_pretest.title', $patient_name, $user_full_name];
    
            $obj = PatientPretest::allowedPatients()->select($select_columns)
                    ->join('patients', 'patient_pretest.patient_id', '=', 'patients.id')
                    ->join('pretests', 'patient_pretest.pretest_id', '=', 'pretests.id')
                    ->join('users', 'patient_pretest.created_by', '=', 'users.id')
                    ->orderBy('form_date','DESC');
                    
            $th_columns = ['form_date', 'title', 'patient_name', 'user_full_name'];


            // to remove th stdClass
            $data['rows'] = (array) json_decode(json_encode($obj->get()->toArray()), true);

            foreach ($th_columns as $db_col) {
                $data['columns'][$db_col] = '<th>' . __('admin/pretests.columns.' . $db_col) . '</th>';
            }

        } else {

            $table = $this->getPretestsTable($request);

            $result = $table->initTable();

            $rows = $result['result']->toArray();

            $data['rows'] = $rows['data'];

            $data['columns'] = $result['columns'];
        }

        $data['_page_title'] = __('admin/pretests.title');

        echo view('admin.printables.list', $data);
    }

}
