<?php

namespace App\Http\Controllers\Admin\Reports;

use App\Http\Controllers\Admin\Controller;
use App\Libraries\Tabler;
use App\Models\Appointment;
use App\Models\Branch;
use App\Models\Treatment;
use App\Models\User;
use App\Models\UserActivityLog;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class AppointmentsReportController extends Controller
{
    public function index(Request $request)
    {

        $table = $this->getAppointmentsTable($request);

        $data = $table->initTable();

        // Override from Tabler
        $data['search_form_action'] = route('admin.reports.appointments');

        $query_string = request()->query();
       
        $args = '';

        if ( $query_string ) {

            $args = '?';

            foreach ( $query_string as $key => $value ) {
                if ( $value ){
                    $args .= $key .'='. $value;
                }
            }

        }

        $data['print_all'] = route('admin.reports.appointments.print',['selection'=>'all']) . $args;
        $data['print_current'] = route('admin.reports.appointments.print',['selection'=>'current']) . $args;

        // for the filters
        //----- Treatments
        $data['treatments'] = Treatment::allowedTreatments()->get();
        $data['selected_treatment'] = ( request()->query('treatments-filter') ) ?: '' ;

        //----- Branches
        $data['branches'] = Branch::allowedBranches()->get();
        $data['selected_branch'] = ( request()->query('branches-filter') ) ?: '' ;

        //----- min date
        $data['min_date'] = Appointment::allowedPatients()->get()->min('appointment_date');
        $data['selected_start_date'] = ( request()->query('start-date') ) ?: $data['min_date'];

        //----- max date
        $data['max_date'] = Appointment::allowedPatients()->get()->max('appointment_date');
        $data['selected_end_date'] = ( request()->query('end-date') ) ?: $data['max_date'];

        $data['filter_link'] = route('admin.reports.appointments') . $args;

        $this->loadView('reports.appointments', 'reports.appointments.title', 'reports.appointments', [], $data,['bs-select/bootstrap-select.min.css','bs-datepicker/dp-bs-4.min.css'], ['bs-select/bootstrap-select.min.js','momentjs/moment.min.js', 'bs-datepicker/dp-bs-4.min.js']);       
    }

    public function print(Request $request, $selection)
    {

        if ($selection == 'all') {

            $obj = Appointment::allowedBranches()
                ->select(['treatments.treatment as treatment_title', DB::raw('count(*) as total')])
                ->join('treatments', 'appointments.treatment_id', '=', 'treatments.id')
                ->groupBy('treatment_title')
                ->orderBy('total', 'DESC')
                ->orderBy('treatment_title', 'ASC');

            $th = ['treatment_title', 'total'];

            // to remove th stdClass
            $data['rows'] = (array) json_decode(json_encode($obj->get()->toArray()), true);

            foreach ($th as $db_col) {

                $data['columns'][$db_col] = '<th>' . __('admin/reports.appointments.columns.' . $db_col) . '</th>';
            }
            
        } else {

            $table = $this->table($request);

            $result = $table->initTable();

            $rows = $result['result']->toArray();

            $data['rows'] = $rows['data'];

            $data['columns'] = $result['columns'];
        }

        $data['_page_title'] = __('admin/reports.appointments.title');

        echo view('admin.reports.printable.general_table', $data);
    }

    private function getAppointmentsTable(Request $request){

        $model = new Appointment();

        $select = $model->appointmentsListColumns();

        if ($request->get('pid')) {

            $patient_id = (int) $request->get('pid');

            if (!in_array($patient_id, allowed_patients())) {
                abort('404');
            }

            $model = $model->where('appointments.patient_id', $patient_id);
        }

        $model = $model->allowedBranches();

        // Apply filters
        $filter_params = request()->query();


        $model = $model->join('treatments', 'appointments.treatment_id', '=', 'treatments.id');

        $model = $model->join('branches', 'appointments.branch_id', '=', 'branches.id');

        $model = $model->join('patients', 'appointments.patient_id', '=', 'patients.id');

        $model = $model->where('appointments.postponed', '=', 0);


        if ( request()->has('treatments-filter') && $filter_params['treatments-filter'] ) {
            $model = $model->where('appointments.treatment_id', '=',  $filter_params['treatments-filter'] );
        }
        
        if ( request()->has('branches-filter') && $filter_params['branches-filter'] ) {
            $model = $model->where('appointments.branch_id', '=',  $filter_params['branches-filter'] );
        }

        // if only start-date is present
        if ( request()->has('start-date') && $filter_params['start-date'] && (!request()->has('end-date') || !$filter_params['end-date'] ) ) {
            $model = $model->where('appointments.appointment_date', '=',  $filter_params['start-date'] );
        }

         // if  start-date and end-date present
         if ( request()->has('start-date') && $filter_params['start-date'] && request()->has('end-date') && $filter_params['end-date']) {

            // flip dates if start date > end date
            if ( $filter_params['start-date'] > $filter_params['end-date'] ){
                $model = $model->whereBetween('appointments.appointment_date', [$filter_params['end-date'],$filter_params['start-date'] ] );
            }else{
                $model = $model->whereBetween('appointments.appointment_date', [$filter_params['start-date'],$filter_params['end-date'] ] );
            }
        }

        // Filter by user branch permissions
        //$model = $model->whereIn('branch_id', allowed_branches());

        //$th = ['ref_number', 'appointment_date', 'patient_name', 'patient_mobile', 'treatment', 'start_time','is_scheduled'];
        $th = ['ref_number', 'appointment_date', 'patient_name', 'patient_mobile', 'treatment', 'start_time','notes'];

        //$sortable = ['ref_number', 'appointment_date', 'patient_name', 'patient_mobile', 'treatment', 'start_time','is_scheduled'];
        $sortable = ['ref_number', 'appointment_date', 'patient_name', 'patient_mobile', 'treatment', 'start_time','notes'];

        $searchable = ['ref_number', 'appointment_date', 'patient_name', 'patient_mobile', 'notes','treatment'];

        if ( is_admin() ||  count(allowed_branches()) > 1) {

            $th[] = 'branch';
            $sortable[] ='branch';
            $searchable[] ='branch';

        }

        $table = new Tabler('admin.reports.appointments', $model, $select, $th, $sortable, $searchable, $request, true);

        return $table;
    }
}
