<?php

namespace App\Http\Controllers\Admin;

use App\Libraries\Tabler;
use App\Models\Branch;
use App\Models\Pretest;
use App\Models\Treatment;
use App\Models\TreatmentCategory;
use App\Rules\GeneralText;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\Rule;

class TreatmentsController extends Controller
{
    public function index(Request $request){

        $table = $this->getTreatmentsTable($request);
       
        $data = $table->initTable();
        
        $this->loadView('treatments.index', 'treatments.title', 'treatments', [], $data);
    }
    //-----------------------------------------
    public function add(Request $request){

        if ( $request->getMethod() == 'POST' ){

            $validated = $request->validate([
                'treatment'  => ['required', 'unique:treatments,treatment', 'min:3', 'max:200', new GeneralText],
                'category' => ['required','exists:treatment_categories,id'],
                'pretests.*' => ['nullable','exists:pretests,id'],
                //'conflict-treatments' => ['nullable'],
                //'conflict-treatments.*' => ['exists:treatments,id'],
                'session-period' => ['required','gte:'. config('admin.min_session_duration'), 'lte:300'],
                'min-sessions' => ['required', 'gte:1', 'lte:25'],
                'branches' => ['required'],
                'branches.*' => ['required','exists:branches,id'],
                'treatment-devices' => ['required'],
                'treatment-devices.*' => ['required','gte:1', 'lte:25'],
                'treatment-prices' => ['required'],
                'treatment-prices.*' => ['required','gte:0', 'lte:100000'],
            ]);

            $insert = [
                'treatment' => $validated['treatment'],
                'min_sessions' => $validated['min-sessions'],
                'session_period' => $validated['session-period'],
                'category_id' => $validated['category'],
            ];

            DB::transaction(function() use ($insert,$validated ) {
                // Add pretests to pivot
                $treatment = Treatment::create($insert);

                if ( $validated['pretests'] ){
                    $treatment->pretests()->attach($validated['pretests']);
                }
                

                // Add conflict to pivot
                /* if ( array_key_exists('conflict-treatments', $_POST) && !empty(array_filter($_POST['conflict-treatments']))){
    
                    $treatment->conflicts()->attach($validated['conflict-treatments']);
                } */

                // insert treatment-branch info
                foreach ( $validated['branches'] as $id => $value){

                    // validation 
                    if ( array_key_exists($id,$validated['treatment-devices']) && array_key_exists($id,$validated['treatment-prices']) ){
                        
                        $details[$id] = [
                            'devices' => $validated['treatment-devices'][$id],
                            'price' => $validated['treatment-prices'][$id],
                        ];
                    }
                }

                $treatment->branches()->attach($details);
                
            });

            
            return redirect()->route('admin.treatments')->with('success', __('admin/common.msgs.success.add'));
            
        }

        $data['treatments'] = Treatment::orderBy('treatment')->get();
        $data['branches'] = Branch::orderBy('branch')->get();
        $data['pretests'] = Pretest::where('for_physicians',0)->orderBy('title')->get();
        $data['categories'] = TreatmentCategory::orderBy('name')->get();

        $this->loadView('treatments.add','treatments.add_title','treatments.add',[],$data,['bs-select/bootstrap-select.min.css'], ['bs-select/bootstrap-select.min.js']);

    }
    //-----------------------------------------
    public function edit(Request $request,Treatment $treatment){

        if ( $request->getMethod() == 'POST'){

            $validated = $request->validate([
                'treatment'  => ['required', Rule::unique('treatments', 'treatment')->ignore($treatment->id), 'min:3', 'max:200', new GeneralText],
                'category' => ['required','exists:treatment_categories,id'],
                'pretests' => ['required'],
                'pretests.*' => ['required','exists:pretests,id'],
                //'conflict-treatments' => ['nullable'],
                //'conflict-treatments.*' => ['nullable','exists:treatments,id'],
                'session-period' => ['required','gte:'. config('admin.min_session_duration'), 'lte:300'],
                'min-sessions' => ['required', 'gte:1', 'lte:25'],
                'branches' => ['required'],
                'branches.*' => ['required','exists:branches,id'],
                'treatment-devices' => ['required'],
                'treatment-devices.*' => ['required','gte:1', 'lte:50'],
                'treatment-prices' => ['required'],
                'treatment-prices.*' => ['required','gte:0', 'lte:100000'],
            ]);

            $insert = [
                'treatment' => $validated['treatment'],
                'category_id' => $validated['category'],
                'min_sessions' => $validated['min-sessions'],
                'session_period' => $validated['session-period'],
            ];

            DB::transaction(function() use ($insert,$treatment,$validated ) {

                Treatment::where('id', $treatment->id)->update($insert);

                $treatment->pretests()->sync($validated['pretests']);

                // Update conflicts
                /* if ( array_key_exists('conflict-treatments', $_POST) && !empty(array_filter($_POST['conflict-treatments']))){

                    $treatment->conflicts()->sync($validated['conflict-treatments']);

                } */

                // insert treatment-branch info
                foreach ( $validated['branches'] as $id => $value){

                    // validation 
                    if ( array_key_exists($id,$validated['treatment-devices']) && array_key_exists($id,$validated['treatment-prices']) ){
                        
                        $details[$id] = [
                            'devices' => $validated['treatment-devices'][$id],
                            'price' => $validated['treatment-prices'][$id],
                        ];
                    }
                }

                $treatment->branches()->sync($details);
            });

            return redirect()->route('admin.treatments')->with('success', __('admin/common.msgs.success.edit'));

        }


        $data['treatment'] = $treatment;
        $data['treatments'] = Treatment::orderBy('treatment')->get();
        $data['branches'] = Branch::orderBy('branch')->get();
        $data['pretests'] = Pretest::where('for_physicians',0)->orderBy('title')->get();
        $data['categories'] = TreatmentCategory::orderBy('name')->get();

        $data['treatment_branches'] = $treatment->branches;

        
        // Get pivot extra fields
        $branches = $treatment->branches;

        $details = [];
        foreach ($branches as $branch) {
            $details[$branch->id] = [
                'devices' =>  $branch->pivot->devices,
                'price' =>  $branch->pivot->price,
            ];
        }

        $data['treatment_details'] = $details;

        $pretests = $treatment->pretests;
        
        $selected_pretests = [];

        if ( $pretests->count() ){
            foreach ($pretests as $pretest) {
                $selected_pretests[] = $pretest->id;
            }
        }

        $data['selected_pretests'] = implode("','", $selected_pretests);

        //--------------------------------------
        $conflicts = $treatment->conflicts;
        
        $selected_conflicts = [];

        if ( $conflicts->count() ){
            foreach ($conflicts as $conflict) {
                $selected_conflicts[] = $conflict->id;
            }
        }
        
        $data['selected_conflicts'] = implode("','", $selected_conflicts);

        //---------------------------------------------

        $this->loadView('treatments.edit', 'treatments.edit_title', 'treatments.edit', $treatment, $data, ['bs-select/bootstrap-select.min.css'], ['bs-select/bootstrap-select.min.js']);


    }
    //-----------------------------------------
    public function delete(Request $request, Treatment $treatment){

        if ( $treatment->appointments && $treatment->appointments->count() > 0 ){
            
            return redirect()->route('admin.treatments')->with('danger', __('admin/common.msgs.error.delete'));

        }

        $treatment->delete();

        return redirect()->route('admin.treatments')->with('success', __('admin/common.msgs.success.delete') );

    }

    /*================================================*/
    private function getTreatmentsTable(Request $request){
        
        $route_key = 'admin.treatments';

        $model = new Treatment();

        $model = $model->allowedTreatments();
        
        $model = $model->join('treatment_categories', 'treatment_categories.id', '=', 'treatments.category_id');

        $th = ['treatment','min_sessions','session_period','cat_name'];
        
        $select = [
            'treatments.id', 
            'treatment',
            'min_sessions',
            DB::raw("CONCAT(session_period,' " . __('admin/common.text.minutes') ."') AS session_period"),
            'treatment_categories.name as cat_name'
        ];
        $searchable = ['treatment','cat_name'];
        $sortable = ['treatment','session_period','min_sessions','cat_name'];

        $table = new Tabler($route_key, $model, $select, $th, $sortable, $searchable, $request,true);

        return $table;
    }
    /*================================================*/
    public function print(Request $request, $selection)
    {
        if ($selection == 'all') {

            $select = [
                'treatments.id', 
                'treatment',
                'min_sessions',
                DB::raw("CONCAT(session_period,' " . __('admin/common.text.minutes') ."') AS session_period"),
                'treatment_categories.name as cat_name'
            ];
           
            $obj = Treatment::allowedTreatments()->select($select)
                ->join('treatment_categories', 'treatment_categories.id', '=', 'treatments.category_id')
                ->orderBy('treatment');


            $th = ['treatment','min_sessions','session_period','cat_name'];


            // to remove th stdClass
            $data['rows'] = (array) json_decode(json_encode($obj->get()->toArray()), true);

            foreach ($th as $db_col) {
                $data['columns'][$db_col] = '<th>' . __('admin/treatments.columns.' . $db_col) . '</th>';
            }
        } else {

            $table = $this->getTreatmentsTable($request);

            $result = $table->initTable();

            $rows = $result['result']->toArray();

            $data['rows'] = $rows['data'];

            $data['columns'] = $result['columns'];
        }

        $data['_page_title'] = __('admin/treatments.title');

        echo view('admin.printables.list', $data);
    }
}
