<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Libraries\AppointmentsCommon;
use App\Models\Appointment;
use Illuminate\Http\Request;
use Laravel\Sanctum\PersonalAccessToken;

class Appointments extends Controller
{
    /**

     * Get Available Appointments

     * @param int $patient_id

     * @param int $branch_id

     * @param int $treatment_id

     * @return void

     */

    public function getAppointmentsCalendar($patient_id, $branch_id, $treatment_id)
    {

        $app_common = new AppointmentsCommon(false);

        $result = $app_common->generateCalendar($patient_id, $branch_id, $treatment_id, -1);

        return response()->json(['appointments' => $result]);
    }

    /**

     * Add new appointments

     *

     * @param Request $request

     * @return void

     */

    public function addAppointment(Request $request)
    {
        if ($request->getMethod() == 'POST') {

            $app_common = new AppointmentsCommon(false);

            $app_common->addAppointment($request);

            return $app_common->getResponse();
        }

        return response()->json(['error' => 'Bad request'], 400);
    }

    //-----------------------------------------------------------
    public function getPatientAppointments(Request $request, $patient_id)
    {

        $user_id = $this->getUserByToken($request);

        //--- if the sent id doesn't match the id from db, 403 forbidden 
        if ($patient_id != $user_id) {

            return response()->json(['error' => '403'], 403);
        }

        $model = new Appointment();

        $result = $model->getPatientAppointments($patient_id);
        
        $appointments = $result->get();

        return response()->json(['appointments' => $appointments], 200);
    }

    //-----------------------------------------------------------

    public function postponeAppointment(Request $request, $appointment_id)
    {
        if ($request->getMethod() == 'POST') {

            $appointment = Appointment::findOrFail($appointment_id);

            if ($appointment && $appointment->id) {

                $patient_id = $appointment->patient_id;

                // prevent the user from seeing other patients

                // info in case of token hijacking

                //--- if the sent id doesn't match the id from db, 403 forbidden 

                $user_id = $this->getUserByToken($request);

                if ($patient_id != $user_id) {

                    return response()->json(['error' => '403'], 403);
                }

                $app_common = new AppointmentsCommon();

                $app_common->changeAppointment($request, $appointment);

                return $app_common->getResponse();
            }

            return response()->json(['error' => '404'], 404);
        }

        return response()->json(['error' => 'Bad request'], 400);
    }



    //-----------------------------------------------------------
    /**

     * Get appointment info

     *

     * @param Request $request

     * @param int $appointment_id

     * @return void

     */

    public function getAppointmentInfo(Request $request, $appointment_id)
    {

        if ($request->getMethod() == 'GET') {

            $appointment = Appointment::findOrFail($appointment_id);

            if ($appointment) {

                $patient_id = $appointment->patient_id;

                // prevent the user from seeing other patients
                // info in case of token hijacking
                //--- if the sent id doesn't match the id from db, 403 forbidden 

                $user_id = $this->getUserByToken($request);

                if ($patient_id != $user_id) {

                    return response()->json(['error' => '403'], 403);
                }

                $model = new Appointment();

                $app = $model->getAppointmentByID($appointment_id);

                return ($app) ? response()->json(['appointment' => $app]) : response()->json(['error' => '404'], 404);
            }

            return response()->json(['error' => '404'], 404);
        }

        return response()->json(['error' => 'Bad request'], 400);
    }

    //-----------------------------------------------------------
    public function cancelAppointment(Request $request, $appointment_id)
    {

        if ($request->getMethod() == 'GET') {

            $appointment = Appointment::findOrFail($appointment_id);

            if ($appointment) {

                $patient_id = $appointment->patient_id;

                // prevent the user from seeing other patients
                // info in case of token hijacking
                //--- if the sent id doesn't match the id from db, 403 forbidden 
                $user_id = $this->getUserByToken($request);

                if ($patient_id != $user_id) {

                    return response()->json(['error' => '403'], 403);
                }

                $app_date_before_update = $appointment->appointment_date;
                $app_start_time_before_update = date('H:i', strtotime($appointment->start_time));
                $today = date('Y-m-d');
                $time = date('H:i');
        
                if ($app_date_before_update < $today || ($app_date_before_update == $today && $app_start_time_before_update < $time)) {
        
                    return response()->json(['error' => __('admin/appointments.error.passed_app')], 400);

                }
        
                if ($appointment->checked_in_at || $appointment->admitted_at) {

                    return response()->json(['error' => __('admin/appointments.error.passed_app')], 400);

                }
        
                $appointment->delete();
        
 
                return response()->json(['success' => 'success'], 200);
            }

            return response()->json(['error' => '404'], 404);
        }

        return response()->json(['error' => 'Bad request'], 400);
    }

    //-----------------------------------------------------------

    private function getUserByToken(Request $request)
    {

        //--- Get the token from request
        $header = $request->header('authorization');

        $token = str_replace('|', '', strstr($header, '|'));

        //--- Get the user id should be using this token

        $hashedToken = hash('sha256', $token);

        $db_token = PersonalAccessToken::where('token', $hashedToken)->first();

        $user = $db_token->tokenable;

        return $user->id;
    }
}
