<?php

namespace App\Http\Controllers\Patient;

use App\Http\Controllers\Patient\Controller;
use App\Libraries\Tabler;
use App\Libraries\AppointmentsCommon;
use App\Models\Appointment;
use App\Models\Branch;
use App\Models\Patient;
use App\Models\Treatment;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Validator;

class AppointmentsController extends Controller
{
    public function index(Request $request){
        
        $model = new Appointment();

        $select = ['ref_number','appointments.id','branches.branch','treatments.treatment','start_time','appointment_date'];

        $model = $model->select($select);

        $model = $model->join('treatments', 'appointments.treatment_id', '=', 'treatments.id');

        $model = $model->join('branches', 'appointments.branch_id', '=', 'branches.id');

        $model = $model->where('appointments.postponed', '=', 0);
        $model = $model->where('appointments.patient_id', '=', patient_id());

        $model = $model->orderBy('appointment_date')->orderBy('start_time');

        $th = ['ref_number', 'appointment_date', 'treatment', 'branch', 'start_time'];

        $sortable = ['ref_number', 'appointment_date', 'treatment', 'branch', 'start_time'];

        $searchable = ['ref_number', 'appointment_date', 'treatment', 'branch'];

        $table = new Tabler('patient.appointments', $model, $select, $th, $sortable, $searchable, $request, true,'patient');

        $data = $table->initTable();

        $data['patient'] = Patient::findOrFail(patient_id());
        
        $this->loadView('appointments.list','appointments.title','appointments',[],$data);
    }

    
    public function add(Request $request){

        if ($request->method() == 'POST' && $request->ajax()) {

            $app_common = new AppointmentsCommon();

            $app_common->addAppointment($request);

            return $app_common->getResponse();
            
        }

        $data['branches'] = Branch::orderby('branch')->get();

        $data['treatments'] = Treatment::orderby('treatment')->get();

        $data['patient'] = Patient::findOrFail(patient_id());

        $this->loadView('appointments.add','appointments.add.title','appointments.add',[],$data,['bs5-select/bootstrap-select.min.css'], ['bs5-select/bootstrap-select.min.js']);

    }

    //-------------------------------------------


    public function edit(Request $request, Appointment $appointment){

        // redirect to appointments if patient_id != patient_id from appointment
        if ( $appointment->patient_id != patient_id() ){

            return  response()->json(['success' => 'success']);
        
        }

        if ( $request->getMethod() == 'POST' && $request->ajax()) {

            $app_common = new AppointmentsCommon();

            $app_common->changeAppointment($request,$appointment);

            return $app_common->getResponse();

        }

        $model = new Appointment();

        $data['appointment'] = $model->getAppointmentByID($appointment->id);

        $data['branches'] = Branch::orderBy('branch')->get();

        $data['treatments'] = Branch::with(['treatments' => function ($q) {
            $q->orderBy('treatment');
        }])->find($appointment->branch_id)->treatments;

        $app_common = new AppointmentsCommon();
        
        $calendar = $app_common->generateCalendar(patient_id(), $appointment->branch_id, $appointment->treatment_id, 0,'patient');

        $calendar['selected_date'] = $appointment->appointment_date;
        $calendar['selected_time'] = date('h:i A', strtotime($appointment->start_time));

        $data['app_table'] = view('patient.appointments.table', $calendar);

        $data['patient'] = Patient::findOrFail(patient_id());

        $this->loadView('appointments.edit', 'appointments.edit.title', 'appointments.edit', $appointment, $data, ['bs5-select/bootstrap-select.min.css'], ['bs5-select/bootstrap-select.min.js']);

    }


    //-------------------------------------------

    public function info(Request $request, Appointment $appointment){


        if ( $request->ajax() && ($appointment->patient_id == patient_id() ) ){

            //$model = new Appointment;

            $data['appointment'] = $appointment;
    
            return response()->view('patient.appointments.info_modal', $data);
        }

    }


    //-------------------------------------------
    public function delete(Request $request, $appointment_id)
    {

        $appointment = Appointment::findOrFail($appointment_id);

        if ( $appointment->patient_id != patient_id() ){
            return redirect()->route('patient.appointments');
        }

        $app_date_before_update = $appointment->appointment_date;
        $app_start_time_before_update = date('H:i', strtotime($appointment->start_time));
        $today = date('Y-m-d');
        $time = date('H:i');

        if ($app_date_before_update < $today || ($app_date_before_update == $today && $app_start_time_before_update < $time)) {

            return redirect()->route('patient.appointments')->with('danger', __('admin/appointments.error.passed_app'));
        }

        if ($appointment->checked_in_at || $appointment->admitted_at) {
            return redirect()->route('patient.appointments')->with('danger', __('admin/appointments.error.passed_app'));
        }

        $review = $appointment->review;

        if ($review && $review->count() > 0){

            return redirect()->route('patient.patients')->with('danger', __('admin/common.msgs.error.delete'));
        
        }

        $appointment->delete();

        return redirect()->route('patient.appointments')->with('success', __('admin/common.msgs.success.delete'));
    }

    //-------------------------------------------
    public function getTreatments(Request $request, int $branch_id = 1)
    {
        if ($request->ajax()) {

            $treatments = Branch::with(['treatments' => function ($q) {
                $q->orderBy('treatment', 'asc');
            }])->find($branch_id)->treatments;

            if ($treatments->count() > 0) {
                return response()->json(['treatments' => $treatments->toArray()]);
            } else {
                return response()->json(['error' => __('admin/common.msgs.no_results')], 404);
            }
        }
    }

    //---------------------------------------------

    public function getCalendar(Request $request, $patient_id, $branch_id, $treatment_id, $week_number)
    {
        
        $patient_id = patient_id();

        if ($request->ajax()) {

            // Validate route variables
            $rules = [
                'branch_id' => ['required', 'exists:branches,id'],
                'treatment_id' => ['required', Rule::exists('branch_treatment', 'treatment_id')->where('branch_id', $branch_id)],
                'week_number' => ['required', 'min:0', 'max:3'],
            ];

            $validator = Validator::make($request->route()->parameters(), $rules);

            if ($validator->fails()) {

                return response()->json(['error' => $validator->errors()->all()]);
            }

            //------------------------------------------------
            
            $app_common = new AppointmentsCommon();
            $data = $app_common->generateCalendar($patient_id, $branch_id, $treatment_id, $week_number,'patient');

            return response()->view('patient.appointments.table', $data);
        }
    }

}
