<?php

namespace App\Libraries;

use App\Rules\GeneralText;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Validator;

class Tabler
{

    private $model; // model object
    private $base_url; // base route
    private $route_key;
    private $request; // request object
    private $th_columns; // 
    private $sortable_columns; // used for validation
    private $searchable_columns;
    private $custom_filters; // controller-specific filters
    private $rows_per_page;

    // results
    private $url; // built url to be used in sort links
    private $rows; // number of rows per page
    private $filter; //selected filter
    private $search; // search keywords
    private $sort; // selected sort_by
    private $order; // selected order (asc or desc)
    private $next_order; // next click's order (asc or desc) 
    private $query_result; // query result object after getQueryResult
    private $use_having;

    private $scope;


    /**
     * Constructor
     *
     * @param string $base_url  the controller route
     * @param Illuminate\Database\Eloquent\Model $model Model object
     * @param array $sortable_columns contains the sortable column names for validation purposes
     * @param Request $request Controller's incomming request
     * @param array $custom_filters any additional filters for this specific controller as action_keyword=>value
     */
    function __construct($route_key, $model, $select_columns = '*', $th_columns, $sortable_columns, array $searchable_columns, Request $request, $use_having = false, $scope = 'admin', $custom_filters = [])
    {


        // set the scope (admin, website, patient,...)
        $this->scope = $scope;

        // set the route key to used with action buttons
        $this->route_key = $route_key;


        // set the base_url
        $this->base_url = route($route_key);


        // Items per page
        if ($this->scope == 'admin') {

            $this->rows_per_page = (array) config('admin.admin_page_rows');

            $this->rows = (int) config('admin.default_admin_page_rows');
        } else {

            $this->rows_per_page = (array) config('website.items_per_page');

            $this->rows = (int) config('website.default_items_per_page');
        }


        //$this->filter = '';

        $this->search = '';
        $this->sort = 'id';
        $this->order = 'desc';
        $this->next_order = 'asc';

        $this->th_columns = $th_columns;

        $this->request = $request;
        $this->sortable_columns = $sortable_columns; // used for validation
        $this->searchable_columns = $searchable_columns;

        /****
         * 
         * Having is used when giving an alias to a table
         * This issue faced me when concatenating columns as alias (patients module)
         * 
         */
        $this->use_having = $use_having;

        // Initialize the query builder
        //---- if selected columns has string value
        if ($select_columns == '*' || empty($select_columns)) {

            // --- if select columns is array and has columns
        } else if (is_array($select_columns)) {

            //$model = $model->select($select_columns[0]);
            foreach ($select_columns as $col) {

                $model = $model->addSelect($col);
            }
        }

        $this->model = $model;


        //$this->custom_filters = $custom_filters; // controller-specific filters        

    }

    //-------------------------------------------------------------

    /**
     * Initialize table parameters
     *
     * @return void
     */
    public function initTable()
    {

        // get rows per page
        $table['rows'] = $this->getRowsPerPage();

        // get filters
        //$table['filter'] = $this->getFilters();

        // get search 
        $table['search'] = $this->getSearch();

        // set sort & order
        $this->setSort();

        // get query result
        $table['result'] = $this->getQueryResult();

        // get pagination info
        //$table['pagination'] = $this->tablePagination();

        $table['columns'] = $this->getTheadColumns();

        $table['route_key'] = $this->route_key;

        $table['base_link'] = $this->base_url;

        $table['search_form_action'] = route($this->route_key);

        return $table;
    }



    //-------------------------------------------------------------

    /**
     * Get rows per page from URL
     *
     * @return string number of rows per page
     */
    public function getRowsPerPage()
    {

        if ($this->request->get('rows') && in_array($this->request->get('rows'), $this->rows_per_page)) {

            $this->rows = (int) $this->request->get('rows');
        }

        $this->url = $this->base_url . '?rows=' . $this->rows;

        return $this->rows;
    }

    //-------------------------------------------------------------

    /**
     * Get selected filter from get request
     *
     * @return string the selected filter action_keyword
     */
    public function getFilters()
    {

        $default_filters = array_keys((array) config('admin.filters'));

        $custom_filters = array_keys($this->custom_filters);

        $filters = array_merge($default_filters, $custom_filters);

        if ($this->request->get('filter') && in_array($this->request->get('filter'), $filters)) {

            $this->filter = (string) $this->request->get('filter');
        }

        $this->url .= '&filter=' . $this->filter;

        // Set the query builder
        switch ($this->filter) {

            case 'enabled':
                $this->model = $this->model->where('is_enabled', true);
                break;
            case 'disabled':
                $this->model = $this->model->where('is_enabled', false);
                break;
            case 'trash':
                $this->model = $this->model->onlyTrashed();
                break;
            default:
                break;
        }

        return $this->filter;
    }

    //---------------------------------------------------------------
    /**
     * Gets the search term fro the URL
     *
     * @return string search keyword
     */
    private function getSearch()
    {

        $rules = [
            'search' => ['nullable', new GeneralText],
        ];

        $validator = Validator::make($this->request->all(), $rules);

        /*if ( $this->request->input('search') && !preg_match('/[\'^£$%&*()}"{@#~?><>,|=_+¬-]/', $this->request->input('search') ) ){*/
        if ($this->request->input('search') && !$validator->fails()) {

            $this->search = (string) Str::limit($this->request->get('search', 50));

            if ($this->use_having) {

                for ($i = 0; $i < count($this->searchable_columns); $i++) {
                    if ($i == 0) {
                        $this->model = $this->model->havingRaw("UPPER(" . $this->searchable_columns[$i] . ") LIKE '%" . $this->search . "%'");
                    } else {
                        $this->model = $this->model->orHavingRaw("UPPER(" . $this->searchable_columns[$i] . ") LIKE '%" . $this->search . "%'");
                    }
                }
            } else {
                for ($i = 0; $i < count($this->searchable_columns); $i++) {
                    if ($i == 0) {
                        $this->model = $this->model->whereRaw("UPPER(" . $this->searchable_columns[$i] . ") LIKE '%" . $this->search . "%'");
                    } else {
                        $this->model = $this->model->orWhereRaw("UPPER(" . $this->searchable_columns[$i] . ") LIKE '%" . $this->search . "%'");
                    }
                }
            }

            $this->url .= '&search=' . $this->search;
        }


        return $this->search;
    }

    //---------------------------------------------------------------

    /**
     * Get the 'sort_by' column &  sort order (asc or desc) from URL and sets the model object
     *
     * @return void
     */
    public function setSort()
    {

        // get Sort
        if ($this->request->get('sort') && in_array($this->request->get('sort'), $this->sortable_columns)) {

            $this->sort = (string) $this->request->get('sort');
        }

        // get order
        if ($this->request->get('order') && in_array($this->request->get('order'), ['asc', 'desc'])) {

            $this->order = strtolower($this->request->get('order'));
        }


        if ($this->order == 'desc') {

            $this->model = $this->model->orderByDesc($this->sort);

            $this->next_order = 'asc';
        } else {

            $this->model = $this->model->orderBy($this->sort);

            $this->next_order = 'desc';
        }
    }

    //---------------------------------------------------------------
    /**
     * Execute the query built by previous filters and actions
     *
     * @return object
     */
    public function getQueryResult()
    {

        $this->query_result = $this->model->paginate($this->rows)->appends(request()->query()); // append query string to pagination links

        return  $this->query_result;
    }

    //---------------------------------------------------------------

    /**
     * Generates a well-formatted table pagination
     *
     * @return array
     */
    public function tablePagination()
    {

        if ($this->scope == 'admin') {
            $str_format = __('admin/common.pagination');
        } else {
            $str_format = __($this->scope.'/common.pagination');
        }

        $first_result = $this->query_result->firstItem();

        $count = $this->query_result->count();

        $total = $this->query_result->total();

        $result['msg'] = sprintf($str_format, $first_result, $count, $total);

        $result['links'] = $this->query_result->links();

        return $result;
    }

    //---------------------------------------------------------------
    public function getTheadColumns()
    {

        foreach ($this->th_columns as $db_col) {

            $lang_key = str_replace($this->scope.'.', '', $this->route_key) . '.columns.' . $db_col;

            if (in_array($db_col, $this->sortable_columns)) {

                $columns[$db_col] = $this->getSortLink($db_col, 'asc', $lang_key);

            } else {

                 $columns[$db_col] = '<th>' . __($this->scope.'/' . $lang_key) . '</th>';
          
            }
        }

        return $columns;
    }

    //---------------------------------------------------------------
    /**
     * Generated sortable columns thead in HTML
     *
     * @param string $db_col DB table column name
     * @param string $order Default order (asc or desc)
     * @param string $lang_key Language key (without admin) for translation
     * @param string $class any extra html classes for th (ex. fit text-left)
     * @return string
     */
    public function getSortLink(string $db_col, string $order, string $lang_key, string $class = '')
    {

        $th = '';

        if ($db_col == $this->sort) {

            $th = '<th data-sort="' . $db_col . '" data-order="' . $this->order . '" class="' . $class . '" sorted>';

            $link = $this->url . '&sort=' . $db_col . '&order=' . $this->next_order;
        } else {

            $th = '<th data-sort="' . $db_col . '" data-order="' . $order . '" class="' . $class . '">';

            $link = $this->url . '&sort=' . $db_col . '&order=' . $order;
        }
        
        // add query string to sort link

        $query_string = request()->query();
       
        $args = '&';

        if ( $query_string ) {

            foreach ( $query_string as $key => $value ) {
                if ( $value ){
                    $args .= $key .'='. $value;
                }
            }

        }

        $link .= $args;

        $th .= '<a href="' . $link . '">';
        $th .= __($this->scope .'/' . $lang_key);
        $th .= '</a>';
        $th .= '</th>';

        return $th;
    }
    //----------------------------------------------------------------------

}
