<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Facades\DB;

class Appointment extends Model
{
    use HasFactory;
    use SoftDeletes;

    protected $table = 'appointments';

    protected $fillable = [
        'ref_number','treatment_id', 'patient_id', 'branch_id', 'start_time', 'end_time', 'appointment_date',
        'physician_id','finished_at','admitted_at','checked_in_at','is_confirmed','scheduled','auto_ended','notes'
    ];
    
    protected $casts = [
        'start_time'  => 'date:h:i A',
        'end_time'  => 'date:h:i A',
    ];

    /**
     * Scope a query to filter by allowed branches only
     *
     * @param  \Illuminate\Database\Eloquent\Builder  $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeAllowedBranches($query)
    {
        return (!is_admin() && user_role() ) ? $query->whereIn('appointments.branch_id', allowed_branches()) : $query;
    }
    /**
     * Scope a query to filter by allowed branches only
     *
     * @param  \Illuminate\Database\Eloquent\Builder  $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeAllowedUsers($query)
    {
        return (!is_admin() && user_role() ) ? $query->whereIn('appointments.physician_id', allowed_users()) : $query;
    }

    /**
     * Scope a query to filter by allowed branches only
     *
     * @param  \Illuminate\Database\Eloquent\Builder  $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeAllowedPatients($query)
    {
        return (!is_admin() && user_role() ) ? $query->whereIn('appointments.branch_id', allowed_branches()) : $query;
    }

    //-----------------------------------------------------------

    /******************
     *  Relations
     *****************/
    /**
     * User Appointments
     *
     * @return Illuminate\Database\Eloquent\Concerns\HasRelationships::belongsToMany
     */
    public function patient()
    {
        return $this->belongsTo(Patient::class);
    }

    //-----------------------------------------------------------
    /**
     * User Appointments
     *
     * @return Illuminate\Database\Eloquent\Concerns\HasRelationships::belongsToMany
     */
    public function treatment(){
        return $this->belongsTo(Treatment::class);
    }
    //-----------------------------------------------------------
    /**
     * User Appointments
     *
     * @return Illuminate\Database\Eloquent\Concerns\HasRelationships::belongsToMany
     */
    public function branch(){
        return $this->belongsTo(Branch::class);
    }
    //-----------------------------------------------------------
    /**
     * User Appointments
     *
     * @return Illuminate\Database\Eloquent\Concerns\HasRelationships::belongsToMany
     */
    public function physician()
    {
        return $this->belongsTo(User::class);
    }
    //-----------------------------------------------------------
    /**
     * User Appointments
     *
     * @return Illuminate\Database\Eloquent\Concerns\HasRelationships::HasOne
     */
    public function review()
    {
        return $this->hasOne(Review::class);
    }
    
    /***************************
     *  Appointments Controller
     **************************/

    public function patientFullName(){
        return DB::raw("CONCAT(COALESCE(patients.first_name,''),' ',COALESCE(patients.second_name,''),' ',COALESCE(patients.third_name,''),' ',COALESCE(patients.last_name,'')) AS patient_name");
    }

    public function appointmentsListColumns(){


        $yes = __('admin/common.text.yes');
        $no = __('admin/common.text.no');

        $scheduled  = DB::raw("IF(scheduled ='0','" . $no . "', '" . $yes . "') AS is_scheduled ");


        $select = ['appointments.id as id','ref_number','appointment_date','start_time','treatments.treatment AS treatment','notes',$this->patientFullName(),'patients.mobile AS patient_mobile',$scheduled];

        if ( is_admin() ||  count(allowed_branches()) > 1) {
            $select[] = 'branches.branch as branch';
        }
        return $select;

    }


    /********************
     *  Other Controller
     *******************/

     /**
      * Get appointments by patient_id
      *
      * @param integer $patient_id
      * @return \Illuminate\Database\Query\Builder
      */
    public function getPatientAppointments($patient_id){

        $select = [
            'ref_number',
            'appointments.id as app_id',
            'appointments.branch_id as branch_id',
            'branches.branch as branch',
            'appointments.treatment_id as treatment_id',
            'treatments.treatment',
            DB::raw('DATE_FORMAT(`appointment_date`, "%Y-%m-%d") as appointment_date'),
            DB::raw('DATE_FORMAT(`appointment_date`, "%a") as ap_day'),
            DB::raw('DATE_FORMAT(`start_time`, "%h:%i %p") as start_time'),
            DB::raw('DATE_FORMAT(`end_time`, "%h:%i %p") as end_time'),
            DB::raw('DATE_FORMAT(`checked_in_at`, "%h:%i %p") as checked_in_at'),
            DB::raw('DATE_FORMAT(`finished_at`, "%h:%i %p") as finished_at'),
            'reviews.rating',
            'reviews.review',
        ];

        $model = DB::table($this->table);

        $model = $model->select($select);

        $model = $model->join('treatments','appointments.treatment_id','=','treatments.id');

        $model = $model->join('branches','appointments.branch_id','=','branches.id');

        $model = $model->leftJoin('reviews','reviews.appointment_id','=','appointments.id');

        $model = $model->where('patient_id','=',$patient_id);

        $model = $model->where('appointments.deleted_at','=',null);
        
        $model = $model->orderBy('appointment_date','desc');

        return $model;
        //return $model->get();
    }
    /**
     * Get appointment by ID
     *
     * @param integer $app_id
     * @return \Illuminate\Support\Collection
     */
    public function getAppointmentByID($app_id){

        $select = [
            'ref_number',
            'appointments.id as app_id',
            'appointments.branch_id as branch_id',
            'branches.branch as branch',
            'appointments.treatment_id as treatment_id',
            'treatments.treatment',
            'appointments.patient_id as patient_id',
            'appointments.notes',
            $this->patientFullName(),
            DB::raw('DATE_FORMAT(`appointment_date`, "%Y-%m-%d") as appointment_date'),
            DB::raw('DATE_FORMAT(`appointment_date`, "%a") as ap_day'),
            DB::raw('DATE_FORMAT(`start_time`, "%h:%i %p") as start_time'),
            DB::raw('DATE_FORMAT(`end_time`, "%h:%i %p") as end_time'),
            DB::raw('DATE_FORMAT(`finished_at`, "%h:%i %p") as finished_at'),
            'reviews.rating as rating',
            'reviews.review as review',
        ];

        $model = DB::table($this->table);

        $model = $model->select($select);

        
        $model = $model->join('treatments','appointments.treatment_id','=','treatments.id');

        $model = $model->join('branches','appointments.branch_id','=','branches.id');

        $model = $model->join('patients','appointments.patient_id','=','patients.id');

        $model = $model->leftJoin('reviews','appointments.review_id','=','reviews.id');

        $model = $model->where('appointments.id','=',$app_id);
        
        /* if ( !is_admin() && user_role() ){

            $model = $model->whereIn('appointments.branch_id',allowed_branches());
        } */
        
        return $model->get()->first();
    }


}
