<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Treatment extends Model
{
    use HasFactory;
    use SoftDeletes;

    protected $fillable = ['treatment','min_sessions','session_period','category_id'];



    /**
     * Scope a query to filter by allowed patients only
     *
     * @param  \Illuminate\Database\Eloquent\Builder  $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeAllowedTreatments($query)
    {

        return (!is_admin() ) ? $query->whereIn('treatments.id', allowed_treatments()) : $query;
        
    }




    //---------------------
    // Eloquent Relations
    //---------------------
    /**
     * Treatment category
     *
     * @return void
     */
    public function category()
    {
        return $this->belongsTo(TreatmentCategory::class);
    }
    //-------------------------------------------------------
    /**
     * Pretests
     *
     * @return Illuminate\Database\Eloquent\Concerns\HasRelationships::belongsToMany
     */
    public function pretests(){
        return $this->belongsToMany(Pretest::class);
    }
    //------------------------------------------------------
    /**
     * Branches
     *
     * @return Illuminate\Database\Eloquent\Concerns\HasRelationships::belongsToMany
     */
    public function branches(){
        return $this->belongsToMany(Branch::class)->withPivot(['devices','price']);
    }
    //------------------------------------------------------
    /**
     * Branches
     *
     * @return Illuminate\Database\Eloquent\Concerns\HasRelationships::belongsToMany
     */
    public function conflicts(){
        return $this->belongsToMany(Treatment::class,'conflict_treatment');
    }
    /**
     * Appointments
     *
     * @return Illuminate\Database\Eloquent\Concerns\HasRelationships::belongsToMany
     */
    public function appointments(){
        return $this->hasMany(Appointment::class,'appointments');
    }


    //--------------------------------------------------------

    /*===================
     *  Custom Functions
     *==================*/

    public function getBranchTreatments($branch_id){

        $treatments = Branch::with(['treatments' => function ($q) {
            $q->orderBy('treatment', 'asc');
        }])->find($branch_id)->treatments;

        return $treatments;
    }

    //---------------------------------------------------------

    public function getPivotFields($treatment_id,$branch_id){

        $treatment = Branch::with(['treatments' => function ($q) use ($treatment_id) {
            $q->where('id', $treatment_id)->first();
        }])->find($branch_id)->treatments;

        if ($treatment->count() != 1) {
            return null;
        }

        // The selected treatment single session duration
        return $treatment->first()->pivot;
    }

}
